<?php

/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */

/*
 * Created on 18. Jul. 2007 by Norman Markgraf (nmarkgraf(at)user.sourceforge.net)
 */

/**
 * TOOL
 * 
 * This is a simple source code beautifier for the OpenDocumentPHP source code.
 * 
 * PHP versions 5
 *   
 * LICENSE:
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the GPL. For more information please see
 * <http://opendocumentphp.org>.
 * 
 * $Id: SourceBeautifier.php 207 2007-07-20 09:17:51Z nmarkgraf $
 * 
 * @category    File Formats
 * @package     OpenDocumentPHP
 * @subpackage  tools_sourcebeautifier
 * @author      Norman Markgraf (nmarkgraf(at)user.sourceforge.net)
 * @copyright   Copyright in 2006, 2007 by The OpenDocumentPHP Team 
 * @license     http://www.gnu.org/licenses/gpl.html GNU General Public License 2.0.
 * @version     SVN: $Id: SourceBeautifier.php 207 2007-07-20 09:17:51Z nmarkgraf $
 * @link        http://opendocumentphp.org
 * @since       0.5.3 - 19. Jul. 2007
 */

/**
 * We need our own Exception class here, because maybe we can not use PEAR packages and link
 * them to a GPL licensed class... 
 */
require_once 'PEAR/Exception.php';

/**
 * SourceBeautifiere class
 * 
 *  
 * @category    File Formats
 * @package     OpenDocumentPHP
 * @subpackage  tools_sourcebeautifier
 * @author      Norman Markgraf (nmarkgraf(at)user.sourceforge.net)
 * @copyright   Copyright in 2006, 2007 by The OpenDocumentPHP Team 
 * @license     http://www.gnu.org/licenses/gpl.html GNU General Public License 2.0.
 * @version     Release: @package_version@
 * @link        http://opendocumentphp.org
 * @since       0.5.3 - 19. Feb. 2007
 */
class SourceBeautifier {
    
    /**
     * 
     * @var string
     * @access private
     */
    private $inputFilename;
    
    /**
     * 
     * @var string
     * @access private
     */
    private $outputFilename;
    
    /**
     * (Default) set of the options.
     * 
     * @var array
     * @access private
     */
    private $options = array(
        'replaceTabsBySpaces'               => true,
        'TabsInSpaces'                      => 4,
        'FunctionsInBSDAllmanStyle'         => true,
        'EndOfLineType'                     => '',
        'insertLinefeedBeforeDocComment'    => true,
        'trimRightside'                     => true
    );
    
    /**
     * 
     * @var string
     * @access private
     */
    private $lineend; 
    
    /**
     * 
     * @param $infile string 
     * @param $outfile string|null 
     * @param $options
     */
    function __construct($infile, $outfile = null, $options = array() )
    {
       $this->setInputFilename($infile);
       $this->setOutputFilename($outfile);
       $this->setOptions($options); 
    }
    
    /**
     * 
     */
    function setInputFilename($filename) 
    {
        $this->inputFilename = $filename;
    }
    
    /**
     * 
     */
    function setOutputFilename($filename) 
    {
        $this->outputFilename = $filename;
    }
    
    /**
     * 
     */
    function setOptions($options) 
    {
        foreach($options as $key => $value) {
            $this->options[$key] = $value;
        }
        switch ($this->options['EndOfLineStyle']) {
            case 'dos':
                $this->lineend = "\x0D\x0A"; 
                break;
		
            case 'unix':
                $this->lineend = "\x0A";
                break;            
                
            case 'mac':
                $this->lineend = "\x0D";
                break;                                
            
            default:
                $this->lineend = "\n";            
        }
    }
    
    /**
     * @param   $str string
     */
    function replaceTabsBySpaces($str) 
    {
        $retStr = $str; 
        
        if ($this->options['replaceTabsBySpaces']) 
        {
            $retStr='';
            $posCounter = $this->options['TabsInSpaces'];
            
            for($i=0, $len = strlen($str); $i<$len; $i++) {
                $c = $str[$i];
                
                if ($c == "\t") {
                    for($j = 0, $l = $posCounter; $j < $l; $j++) {
                        $retStr .= ' ';			
                    }
                    $posCounter=$this->options['TabsInSpaces'];
                } else {
                    $retStr .= $c;
                    $posCounter--;
                }
                
                if ($posCounter <= 0) {
                    $posCounter = $this->options['TabsInSpaces'];
                }
            }
        } 
        
        return $retStr;
    }
    
    /**
     * 
     * @param   $str string
     */
    function trimRightside($str) 
    {
        $retStr = $str;
        
        if ($this->options['trimRightside']) {
            $retStr = rtrim($str);
        }
        
        return $retStr;
    }
    
    /**
     * 
     * @access  private
     * @param   $str string
     */
    private function _doBSDAllmanStyle($str, $type)
    {
        if (ereg('^([ |\t]*)([abstract |private |public |protected |static |final ]*)'.$type.' .*{$', $str)) {
            $leadingwhite = '';
            $i = 0;
            $strlength = strlen($str);
            
            while ($i < $strlength && ($str[$i] == ' ' || $str[$i] == '\t')) {
                $leadingwhite .= $str[$i];
                $i++;
            }
        
            $str = substr($str, 0, -1);     
            $str = $str . $this->lineend . $leadingwhite .'{'; 
        }                
        return $str;
    }
    
    /**
     * 
     * @param   $str string
     */     
    function correctFunctionBSDAllmanStyle($str) 
    {
        return $this->_doBSDAllmanStyle($str, 'function');
    }
    
    /**
     * 
     * @param   $str string
     */
    function correctClassBSDAllmanStyle($str) 
    {
        return $this->_doBSDAllmanStyle($str, 'class');
    }
    
    /**
     * 
     * @param   $str string
     */
    function correctInterfaceBSDAllmanStyle($str) 
    {
        return $this->_doBSDAllmanStyle($str, 'interface');
    }
    
    /**
     * 
     * @access  private
     * @param   $str string
     * @param   $type string
     */    
    private function _doCorrectIncludeRequireTags($str, $type)
    {
        if (ereg('^([ |\t)*)'.$tag.'\(.*\)', $str)) {
            $str = ereg_replace('^([ |\t)*)'.$tag.'((.)*)\((.*)\)', '\\1'.$tag.' \\2;',$str);
        }
        
        return $str;
    }
    
    /**
     * 
     * @param   $str string
     */
    function correctIncludeRequireTags($str)
    {
        $str = $this->_doCorrectIncludeRequireTags($str, 'require_once');
        $str = $this->_doCorrectIncludeRequireTags($str, 'include_once');                
    }
    
    /**
     * 
     * @param   $str string
     */
    function linefeedBeforeDocComment($str)
    {
        if ($this->options['insertLinefeedBeforeDocComment']) {
    
            static $wasWhiteline;
	
            if (ereg('^([ |\t]*)\/\*\*([.]*)', $str)) {
                if (!$wasWhiteline) {
                    $str = $this->lineend.$str;
                }
            } else {
                $wasWhiteline = strlen(trim($str)) == 0;
            }
        }             
        return $str;
    }
    
    /**
     * 
     */
    function doFile() 
    {
        try {
            $inStream = $this->openInputfile();
            $outStream = $this->openOutputfile();        
        
            while (!feof($inStream)) {
                $str = fgets($inStream);
                $str = $this->replaceTabsBySpaces($str);
                $str = $this->trimRightside($str);
                $str = $this->correctFunctionBSDAllmanStyle($str);                  
                $str = $this->correctClassBSDAllmanStyle($str);                  
                $str = $this->correctInterfaceBSDAllmanStyle($str);                  
		        $str = $this->linefeedBeforeDocComment($str);
                fputs($outStream, $str. $this->lineend);                                                
            }
            
            fclose($outStream);
            fclose($inStream);
            
        } catch (PEAR_Exception $e) {
            print $e;            
        }
        
    }
    
    /**
     * 
     */
    function openInputfile()
    {
        $inStream = @fopen($this->inputFilename, 'r');
        if (!inStream) {
        
            if (!file_exists($filename)) {
                throw new PEAR_Exception("FATAL ERROR: Input file '.$inputFilename.' does not exists.");
            }
            
            if (!is_readable($filename)) {
                throw new PEAR_Exception("FATAL ERROR: Input file '.$inputFilename.' is not readable.'");
            }
        }
        return $inStream;
    }
    
    /**
     * 
     */
    function openOutputfile()
    {
        $filename = $this->outputFilename;
        
        if (is_null($filename)) {
            $filename = 'php://output';
            $outStream = @fopen($filename, 'w');
        } else {
            $outStream = @fopen($filename, 'w+');
            
            if (!outStream) {
            
                if (file_exists($filename)) {
                    throw new PEAR_Exception("FATAL ERROR: Output file '.$outfile.' exists.");
                }
                
                if (!is_writeable($filename)) {
                    throw new PEAR_Exception("FATAL ERROR: Output file '.$filename.' is not writeable.'");
                }
            }
        }
        
        if (!outStream) {
            throw new PEAR_Exception("FATAL ERROR: Unknow error while trying to open '.$outStream.' as output file.");
        }
        
        return $outStream;
    }    
}

/*
 *  
 */
$filename = '/home/o/opendocumentphp.org/files/opendocumentphp/phase1.1/src/OpenDocumentPHP/OpenDocumentText.php';
$options = array('TabsInSpaces' => 8);

//$app = new sb($filename, $options);
$app = new SourceBeautifier($filename);
$app->doFile();

?>